#!/usr/bin/env python3
"""
EzEpoch SSH Key Generator v1.0
Simple cross-platform GUI to generate SSH keys for cloud GPU access.

ARC Technologies LLC - Patent Pending
"""

import os
import sys
import platform
import subprocess
from pathlib import Path

# Try to import tkinter for GUI
try:
    import tkinter as tk
    from tkinter import ttk, messagebox, filedialog
    HAS_GUI = True
except ImportError:
    HAS_GUI = False

__version__ = "1.0.0"
APP_NAME = "EzEpoch SSH Key Generator"


def get_default_ssh_dir():
    """Get the default .ssh directory for the current platform."""
    if platform.system() == "Windows":
        return Path(os.environ.get("USERPROFILE", "")) / ".ssh"
    else:
        return Path.home() / ".ssh"


def generate_ssh_key(key_name="ezepoch_key", key_type="ed25519", passphrase="", save_dir=None):
    """
    Generate an SSH key pair.
    
    Args:
        key_name: Name for the key files (without extension)
        key_type: Key type (ed25519 recommended, rsa as fallback)
        passphrase: Optional passphrase for the private key
        save_dir: Optional custom directory to save keys (default: ~/.ssh)
    
    Returns:
        tuple: (success, public_key_content, private_key_path, error_message)
    """
    ssh_dir = Path(save_dir) if save_dir else get_default_ssh_dir()
    
    # Create .ssh directory if it doesn't exist
    try:
        ssh_dir.mkdir(parents=True, exist_ok=True)
        # Set proper permissions on Unix
        if platform.system() != "Windows":
            os.chmod(ssh_dir, 0o700)
    except Exception as e:
        return False, None, None, f"Failed to create .ssh directory: {e}"
    
    private_key_path = ssh_dir / key_name
    public_key_path = ssh_dir / f"{key_name}.pub"
    
    # Check if key already exists
    if private_key_path.exists():
        # Read existing public key for user
        if public_key_path.exists():
            existing_pub = public_key_path.read_text().strip()
            return False, existing_pub, str(private_key_path), f"KEY_EXISTS: Key already exists at: {private_key_path}\n\nYour existing public key is shown below - you can use this one!\n\nOr choose a different name to create a new key."
        return False, None, None, f"Key already exists: {private_key_path}\nPlease choose a different name or delete the existing key."
    
    # Build ssh-keygen command
    cmd = ["ssh-keygen"]
    
    if key_type == "ed25519":
        cmd.extend(["-t", "ed25519"])
    else:
        cmd.extend(["-t", "rsa", "-b", "4096"])
    
    cmd.extend([
        "-f", str(private_key_path),
        "-N", passphrase,  # Empty string for no passphrase
        "-C", f"ezepoch-{key_name}@{platform.node()}"
    ])
    
    try:
        # Run ssh-keygen
        result = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=30
        )
        
        if result.returncode != 0:
            return False, None, None, f"ssh-keygen failed: {result.stderr}"
        
        # Read the public key
        if public_key_path.exists():
            public_key = public_key_path.read_text().strip()
            
            # Set proper permissions on Unix
            if platform.system() != "Windows":
                os.chmod(private_key_path, 0o600)
                os.chmod(public_key_path, 0o644)
            
            return True, public_key, str(private_key_path), None
        else:
            return False, None, None, "Public key file was not created"
            
    except FileNotFoundError:
        return False, None, None, "ssh-keygen not found. Please install OpenSSH.\n\nWindows: Settings > Apps > Optional Features > OpenSSH Client\nMac/Linux: Usually pre-installed"
    except subprocess.TimeoutExpired:
        return False, None, None, "ssh-keygen timed out"
    except Exception as e:
        return False, None, None, f"Error running ssh-keygen: {e}"


class SSHKeyGenApp:
    """Simple GUI for SSH key generation."""
    
    def __init__(self):
        self.root = tk.Tk()
        self.root.title(f"{APP_NAME} v{__version__}")
        self.root.geometry("650x580")
        self.root.resizable(True, True)
        
        # Set dark theme colors
        self.bg_color = "#1a1a2e"
        self.fg_color = "#eaeaea"
        self.accent_color = "#00b4d8"
        self.success_color = "#10b981"
        self.card_bg = "#16213e"
        
        self.root.configure(bg=self.bg_color)
        
        # Default save location
        self.save_dir = get_default_ssh_dir()
        
        self.setup_ui()
        
    def setup_ui(self):
        """Build the user interface."""
        # Main container
        main_frame = tk.Frame(self.root, bg=self.bg_color, padx=20, pady=20)
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # Header
        header = tk.Label(
            main_frame,
            text="SSH Key Generator",
            font=("Segoe UI", 24, "bold"),
            fg=self.accent_color,
            bg=self.bg_color
        )
        header.pack(pady=(0, 5))
        
        subtitle = tk.Label(
            main_frame,
            text="Generate SSH keys for secure cloud GPU access",
            font=("Segoe UI", 11),
            fg="#888",
            bg=self.bg_color
        )
        subtitle.pack(pady=(0, 20))
        
        # Key Name Section
        name_frame = tk.Frame(main_frame, bg=self.card_bg, padx=15, pady=15)
        name_frame.pack(fill=tk.X, pady=5)
        
        tk.Label(
            name_frame,
            text="Key Name:",
            font=("Segoe UI", 11),
            fg=self.fg_color,
            bg=self.card_bg
        ).pack(anchor=tk.W)
        
        self.key_name_var = tk.StringVar(value="ezepoch_key")
        name_entry = tk.Entry(
            name_frame,
            textvariable=self.key_name_var,
            font=("Segoe UI", 12),
            bg="#0f0f1a",
            fg=self.fg_color,
            insertbackground=self.fg_color,
            relief=tk.FLAT,
            width=40
        )
        name_entry.pack(fill=tk.X, pady=(5, 0))
        
        # Save Location Section
        loc_frame = tk.Frame(main_frame, bg=self.card_bg, padx=15, pady=15)
        loc_frame.pack(fill=tk.X, pady=5)
        
        tk.Label(
            loc_frame,
            text="Save Location:",
            font=("Segoe UI", 11),
            fg=self.fg_color,
            bg=self.card_bg
        ).pack(anchor=tk.W)
        
        loc_row = tk.Frame(loc_frame, bg=self.card_bg)
        loc_row.pack(fill=tk.X, pady=(5, 0))
        
        self.save_dir_var = tk.StringVar(value=str(self.save_dir))
        self.loc_entry = tk.Entry(
            loc_row,
            textvariable=self.save_dir_var,
            font=("Segoe UI", 10),
            bg="#0f0f1a",
            fg=self.fg_color,
            insertbackground=self.fg_color,
            relief=tk.FLAT
        )
        self.loc_entry.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=(0, 10))
        
        browse_btn = tk.Button(
            loc_row,
            text="Browse...",
            font=("Segoe UI", 9),
            bg="#2563eb",
            fg="white",
            relief=tk.FLAT,
            cursor="hand2",
            command=self.browse_location,
            padx=10
        )
        browse_btn.pack(side=tk.RIGHT)
        
        # Key Type Section
        type_frame = tk.Frame(main_frame, bg=self.card_bg, padx=15, pady=15)
        type_frame.pack(fill=tk.X, pady=5)
        
        tk.Label(
            type_frame,
            text="Key Type:",
            font=("Segoe UI", 11),
            fg=self.fg_color,
            bg=self.card_bg
        ).pack(anchor=tk.W)
        
        self.key_type_var = tk.StringVar(value="ed25519")
        
        type_options = tk.Frame(type_frame, bg=self.card_bg)
        type_options.pack(fill=tk.X, pady=(5, 0))
        
        tk.Radiobutton(
            type_options,
            text="ED25519 (Recommended - Modern & Fast)",
            variable=self.key_type_var,
            value="ed25519",
            font=("Segoe UI", 10),
            fg=self.fg_color,
            bg=self.card_bg,
            selectcolor=self.card_bg,
            activebackground=self.card_bg,
            activeforeground=self.accent_color
        ).pack(anchor=tk.W)
        
        tk.Radiobutton(
            type_options,
            text="RSA 4096 (Legacy compatibility)",
            variable=self.key_type_var,
            value="rsa",
            font=("Segoe UI", 10),
            fg=self.fg_color,
            bg=self.card_bg,
            selectcolor=self.card_bg,
            activebackground=self.card_bg,
            activeforeground=self.accent_color
        ).pack(anchor=tk.W)
        
        # Generate Button
        self.generate_btn = tk.Button(
            main_frame,
            text="Generate SSH Key",
            font=("Segoe UI", 14, "bold"),
            bg=self.accent_color,
            fg="white",
            relief=tk.FLAT,
            cursor="hand2",
            command=self.generate_key,
            padx=30,
            pady=12
        )
        self.generate_btn.pack(pady=20)
        
        # Result Section
        self.result_frame = tk.Frame(main_frame, bg=self.card_bg, padx=15, pady=15)
        
        tk.Label(
            self.result_frame,
            text="Your Public Key (copy this to EzEpoch):",
            font=("Segoe UI", 11, "bold"),
            fg=self.success_color,
            bg=self.card_bg
        ).pack(anchor=tk.W)
        
        self.public_key_text = tk.Text(
            self.result_frame,
            height=4,
            font=("Consolas", 10),
            bg="#0f0f1a",
            fg=self.fg_color,
            wrap=tk.WORD,
            relief=tk.FLAT
        )
        self.public_key_text.pack(fill=tk.X, pady=(10, 10))
        
        button_frame = tk.Frame(self.result_frame, bg=self.card_bg)
        button_frame.pack(fill=tk.X)
        
        self.copy_btn = tk.Button(
            button_frame,
            text="Copy to Clipboard",
            font=("Segoe UI", 10),
            bg="#2563eb",
            fg="white",
            relief=tk.FLAT,
            cursor="hand2",
            command=self.copy_to_clipboard,
            padx=15,
            pady=5
        )
        self.copy_btn.pack(side=tk.LEFT)
        
        self.path_label = tk.Label(
            self.result_frame,
            text="",
            font=("Segoe UI", 9),
            fg="#888",
            bg=self.card_bg,
            wraplength=550
        )
        self.path_label.pack(anchor=tk.W, pady=(10, 0))
        
        # Footer
        footer = tk.Label(
            main_frame,
            text="2025 ARC Technologies LLC | Patent Pending",
            font=("Segoe UI", 9),
            fg="#555",
            bg=self.bg_color
        )
        footer.pack(side=tk.BOTTOM, pady=(20, 0))
        
    def browse_location(self):
        """Let user choose save location."""
        folder = filedialog.askdirectory(
            title="Choose Save Location",
            initialdir=self.save_dir_var.get()
        )
        if folder:
            self.save_dir_var.set(folder)
    
    def generate_key(self):
        """Handle key generation."""
        key_name = self.key_name_var.get().strip()
        if not key_name:
            messagebox.showerror("Error", "Please enter a key name")
            return
        
        # Sanitize key name
        key_name = "".join(c for c in key_name if c.isalnum() or c in "_-")
        
        # Get save directory
        save_dir = Path(self.save_dir_var.get().strip())
        
        self.generate_btn.config(state=tk.DISABLED, text="Generating...")
        self.root.update()
        
        success, public_key, private_path, error = generate_ssh_key(
            key_name=key_name,
            key_type=self.key_type_var.get(),
            save_dir=save_dir
        )
        
        if success:
            self.result_frame.pack(fill=tk.X, pady=10)
            self.public_key_text.delete(1.0, tk.END)
            self.public_key_text.insert(tk.END, public_key)
            self.path_label.config(text=f"Private key saved to: {private_path}")
            self.generate_btn.config(text="Key Generated!", bg=self.success_color)
            
            messagebox.showinfo(
                "Success!",
                f"SSH key generated successfully!\n\n"
                f"Private key: {private_path}\n"
                f"Public key: {private_path}.pub\n\n"
                f"Copy the public key and paste it into EzEpoch."
            )
        elif error and error.startswith("KEY_EXISTS:") and public_key:
            # Key already exists - show the existing public key
            self.result_frame.pack(fill=tk.X, pady=10)
            self.public_key_text.delete(1.0, tk.END)
            self.public_key_text.insert(tk.END, public_key)
            self.path_label.config(text=f"Existing key at: {private_path}")
            self.generate_btn.config(state=tk.NORMAL, text="Generate SSH Key")
            
            messagebox.showinfo(
                "Existing Key Found",
                f"An SSH key already exists!\n\n"
                f"Location: {private_path}\n\n"
                f"Your existing public key is displayed below.\n"
                f"You can copy and use it, or change the key name to create a new one."
            )
        else:
            self.generate_btn.config(state=tk.NORMAL, text="Generate SSH Key")
            messagebox.showerror("Error", error)
    
    def copy_to_clipboard(self):
        """Copy public key to clipboard."""
        public_key = self.public_key_text.get(1.0, tk.END).strip()
        if public_key:
            self.root.clipboard_clear()
            self.root.clipboard_append(public_key)
            self.copy_btn.config(text="Copied!", bg=self.success_color)
            self.root.after(2000, lambda: self.copy_btn.config(text="Copy to Clipboard", bg="#2563eb"))
    
    def run(self):
        """Start the application."""
        self.root.mainloop()


def cli_mode():
    """Run in CLI mode if no GUI available."""
    print(f"\n{'='*50}")
    print(f"  {APP_NAME} v{__version__}")
    print(f"  ARC Technologies LLC - Patent Pending")
    print(f"{'='*50}\n")
    
    key_name = input("Enter key name (default: ezepoch_key): ").strip() or "ezepoch_key"
    key_name = "".join(c for c in key_name if c.isalnum() or c in "_-")
    
    print("\nKey types:")
    print("  1. ED25519 (Recommended)")
    print("  2. RSA 4096")
    choice = input("Choose type (1 or 2, default: 1): ").strip() or "1"
    key_type = "ed25519" if choice == "1" else "rsa"
    
    print(f"\nGenerating {key_type.upper()} key: {key_name}...")
    
    success, public_key, private_path, error = generate_ssh_key(
        key_name=key_name,
        key_type=key_type
    )
    
    if success:
        print(f"\n{'='*50}")
        print("SUCCESS! SSH key generated.")
        print(f"{'='*50}")
        print(f"\nPrivate key: {private_path}")
        print(f"Public key:  {private_path}.pub")
        print(f"\n{'='*50}")
        print("YOUR PUBLIC KEY (copy this to EzEpoch):")
        print(f"{'='*50}")
        print(public_key)
        print(f"{'='*50}\n")
    else:
        print(f"\nERROR: {error}\n")
        sys.exit(1)


def main():
    """Main entry point."""
    print(f"Starting {APP_NAME} v{__version__}...")
    
    if "--cli" in sys.argv:
        cli_mode()
    elif HAS_GUI:
        try:
            print("Launching GUI...")
            app = SSHKeyGenApp()
            app.run()
        except Exception as e:
            print(f"GUI Error: {e}")
            print("Falling back to CLI mode...")
            cli_mode()
    else:
        print("No GUI available, using CLI mode...")
        cli_mode()


if __name__ == "__main__":
    try:
        main()
    except Exception as e:
        print(f"Error: {e}")
        input("Press Enter to exit...")

